/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * 
 * SD/SDIO Driver for the dm388 board.
 * 
 */
 
#include "stdio.h"
#include "dm388_types.h"
#include "sdmmc.h"
#include "gpio.h"

#define WLAN_EN_GPIO     6

UINT32 readl(UINT32 *addr)
{
	return (*(UINT32 *) addr);
}

void writel(UINT32 data, UINT32 *addr)
{
	*(UINT32 *) addr = data;
}

/* Enables the wlan chip by sending by 
 * driving wlan_enable pin high
 */
void enable_wlanchip ()
{
	UINT32 regval;
	
	/* WLAN_EN_GPIO is GPIO6 in Bank 2 */
	
	/* Reset GPIO Subsystem */
	GPIO2_SYSCONFIG = 0x00000020; /* Software Reset */
	DM388_usecDelay(0x90000);
	GPIO2_SYSCONFIG = 0x100; /* no-idle */
	
	/* Output Enable GPIO6 in Bank 2 */		 
	regval = GPIO2_OE;
	regval = regval & ~(1 << WLAN_EN_GPIO);  
	GPIO2_OE = regval;

	/* Set WLAN_ENABLE High */
	GPIO2_DATAOUT |= (1 << WLAN_EN_GPIO);
	GPIO2_SETDATAOUT |= (1 << WLAN_EN_GPIO);
	DM388_usecDelay(100000);
	
	/* Set WLAN_ENABLE Low */
	regval = GPIO2_DATAOUT;
	regval = regval & ~(1 << WLAN_EN_GPIO);
	GPIO2_DATAOUT = regval;
	regval = GPIO2_SETDATAOUT;
	regval = regval & ~(1 << WLAN_EN_GPIO);
	GPIO2_SETDATAOUT = regval;
	DM388_usecDelay(100000);
#ifdef PRINTF
	platform_write ("\tTrying to enable wlan chip\n");
#endif
	/* Set WLAN_ENABLE High */
	GPIO2_DATAOUT |= (1 << WLAN_EN_GPIO);
	GPIO2_SETDATAOUT |= (1 << WLAN_EN_GPIO);
	
	/* Wait for chip to settle */
    DM388_usecDelay(200000);
        
    platform_write("\nWLAN Chip Enabled\n");
}

/* dm388 Board Initialisation */
void dm388_board_init(hsmmc_t *mmc_base)
{
	/* Pin Muxing done in the Gel File */	
	switch ((UINT32) mmc_base) {
	case HSMMC_BASE0:
		enable_wlanchip();
		break;
	case HSMMC_BASE1:
		break;
	}
}

/*
 * Enables mmc clock
 * 
 * Parameters:
 * mmc_base:	Base Address of the MMC controller
 * clock:       Clock Frequency
 * 
 * Returns:
 *   -1			On Failure
 *    0         On Success
 */
INT32 set_mmc_clock(hsmmc_t *mmc_base, UINT32 clock)
{
	UINT32 val;
	UINT32 regval;
	
	/* disables, cen and set data timeout value */
	regval = readl(&mmc_base->sysctl);
	regval &= ~(0xF0007); 
	regval |= (0xe << 16); 

	switch (clock) {
	case CLK_INITSEQ:
		val = (SDMMC_REF_CLK * 1000 / (160) / 2);
		break;
	case CLK_400KHZ:	
		val = (SDMMC_REF_CLK * 1000) / (1600); /* 1.6MHZ */
		break;
	case CLK_12MHZ:
		val = (SDMMC_REF_CLK * 1000 / (12000));
	    break;
	case CLK_24MHZ:
		val = (SDMMC_REF_CLK * 1000 / (24000));
		break;
	default:
		platform_write ("\nClock Frequency not supported\n");
		return FAILURE;
	}

	/* Select clock frequency */
	regval &= ~(0xffc0);
	regval |= (val << 6);

	regval |= (0x1); /* enable clock */
	writel(regval, &mmc_base->sysctl);
	DM388_usecDelay(4000);
	
	/* Wait until stable? */
	while ((readl(&mmc_base->sysctl) & 0x2) == 0x0);

	writel(readl(&mmc_base->sysctl) | 0x4, &mmc_base->sysctl);
	DM388_usecDelay(4000);

	return SUCCESS;
}

/*
 * Initialise the SD/SDIO Controller
 * 
 * Parameters:
 * mmc_base:	Base Address of the SD/SDIO controller
 * 
 * Returns:		void
 * 
 */ 
void mmcinit (hsmmc_t *mmc_base)
{
	UINT32 regval;

	dm388_board_init(mmc_base);
	
	/* software reset */
	regval = readl(&mmc_base->sysconfig);
	writel(regval | 0x2, &mmc_base->sysconfig);

	/* reset done? */
	while ((readl(&mmc_base->sysstatus) & 0x1) == 0);

	/* reset the host controller */
	regval = readl(&mmc_base->sysctl);
	regval |= (1 << 24); /* SRA */
	writel(regval, &mmc_base->sysctl); 
	
	/* reset done? */
	while ((readl(&mmc_base->sysctl) & (1 << 24)) != 0x0);

	/* Support for 1.8V and 3V */
	//writel(0x6000000, &mmc_base->capa);
	writel(0x6E10080, &mmc_base->capa);
	
	/* 1 Bit Mode, 3V Bus Voltage (SVDS) and Bus Pwr Off */
	writel(0x00000C00, &mmc_base->hctl);
	
	/* Set Block Size to 512 Bytes */
	writel(0x200, &mmc_base->blk);
	
	/* Support for 1.8V, 3V cards */
	writel(readl(&mmc_base->capa) | 0x6000000, &mmc_base->capa);

	/* Functional Mode, No INIT Sequence */
	regval = readl(&mmc_base->con) & (0x3 << 9);
	writel(regval, &mmc_base->con);
	DM388_usecDelay(10);

	set_mmc_clock(mmc_base, CLK_INITSEQ);
	
	/* Switch on Bus Power */
	writel(readl(&mmc_base->hctl) | (1 << 8), &mmc_base->hctl); 

	/* Enable Interrupts */
	writel(0x307F0033, &mmc_base->ie);
	
	/* Send Initialisation Sequence for 80 clock cycles */
	writel(0x00000602, &mmc_base->con);
	DM388_usecDelay(10);
	
	/* Send CMD0 */
	writel(SDMMC_CMD0, &mmc_base->cmd);
	while( (readl(&mmc_base->stat) & 0x1) == 0);
	writel(0x1, &mmc_base->stat);

	/* Send CMD0 */
	writel(SDMMC_CMD0, &mmc_base->cmd);
	while( (readl(&mmc_base->stat) & 0x1) == 0);
	writel(0x1, &mmc_base->stat);
	
	/* End Init sequence */
	regval = readl(&mmc_base->con) & ~0x2;
	writel(regval, &mmc_base->con);
}

/*
 * Send a command
 * 
 * Parameters:
 * mmc_base:	Base Address of the SD/SDIO controller
 * cmd:			command
 * arg:			arguments to the command
 * 
 * Returns:		
 * 	0 		On Success
 *  > 0     On Failure (Contents of HS_STAT register)
 *  
 */ 
INT32 send_cmd(hsmmc_t * mmc_base, UINT32 cmd, UINT32 arg)
{
	UINT32 regval;
	UINT32 status;

	/* Software reset cmd and dat lines */
	regval = readl(&mmc_base->sysctl);
	regval |= 0x6000000;
	writel(regval, &mmc_base->sysctl);

	while( (readl(&mmc_base->sysctl) & 0x6000000) != 0);

	/* wait until you are allowed to send cmd */
	while ((readl(&mmc_base->pstate) & 0x2) == 0x2);
	
	writel(0x307F0037, &mmc_base->stat);
	writel(0x200, &mmc_base->blk);
	
	writel(arg, &mmc_base->arg);
	writel(cmd, &mmc_base->cmd);
	DM388_usecDelay(0x1000);

	for ( ;; ) {
		do {
			status = readl(&mmc_base->stat);
		} while (status == 0);

		if (status & STAT_ERRI)
			return status;
		
		if (status & STAT_CC) { /* command complete */
			writel(STAT_CC, &mmc_base->stat); 
			return SUCCESS;
		}
	}
}

/*
 * Detect an SDIO/SD Card
 * 
 * Parameters:
 * mmc_base:			Base Address of the SD/SDIO controller
 * high_capacity_card:	Returns 1 if a High Capacity SD card is found.
 * 
 * Returns:	The Type of Card
 * 
 *  (1):	SDIO Card is Found
 *  (2):	SD Card is Found (SD_CARD)
 *  (3):	MMC Card is Found
 * (-1):	Card Not Found (UNKNOWN_CARD) 
 * 
 */ 
INT32 detect_card (hsmmc_t *mmc_base, INT32 * high_capacity_card)
{
	UINT32 ocr, ocr_rcvd;
	UINT32 rca;
	INT32 cardtype = UNKNOWN_CARD;
	INT32 err;
	INT32 retry;
	INT32 io_funcs;
	INT32 ver2_card = 0;

	set_mmc_clock(mmc_base, CLK_400KHZ);
	send_cmd(mmc_base, SDMMC_CMD0, 0x0000);
	DM388_wait(0x1fff);
	err = send_cmd(mmc_base, SDMMC_CMD5, 0x0000);
	if (!err) {
		ocr_rcvd =  readl(&mmc_base->rsp10); 
		io_funcs = (ocr_rcvd >> 28) & 0x7;
		
		do {
			err = send_cmd(mmc_base, SDMMC_CMD5, ((ocr_rcvd >> 8) & 0xffffff));		
			if (err)
				return -err;
	
		} while ( (readl(&mmc_base->rsp10) & 0x80000000) == 0); /* Is card Ready? */
		
		platform_write ("\nDetected an SDIO Card with %d functions\n", io_funcs);
		cardtype = SDIO_CARD;
	} else {
		/* Re-Init Card */
		writel(SDMMC_CMD0, &mmc_base->cmd);
		while( (readl(&mmc_base->stat) & 0x1) == 0);
		writel(0x1, &mmc_base->stat);
	
		err = send_cmd(mmc_base, SDMMC_CMD8, 0x000001AA);
		if (!err)
			ver2_card = 1;
		else
			ver2_card = 0;

		ocr = 0x1FF << 15;
		err = send_cmd(mmc_base, SDMMC_CMD55, 0x00000000);
		if (!err) {
			cardtype = SD_CARD;
			ocr = ocr | (ver2_card << 30);
			err = send_cmd(mmc_base, SDMMC_ACMD41, ocr);
		} else {
			cardtype = MMC_CARD;
			err = send_cmd(mmc_base, SDMMC_CMD1, ocr);
		}

		if (err)
			return -err;

		ocr_rcvd =  readl(&mmc_base->rsp10);
		if (ver2_card && (ocr_rcvd & 0x40000000))
			*high_capacity_card = 1;
		else
			*high_capacity_card = 0;

		/* is card is ready? */
		for (retry = 0; retry < 1000 && ((ocr_rcvd & 0x80000000) == 0); retry++) {
			if (cardtype == SD_CARD)
				send_cmd(mmc_base, SDMMC_CMD55, 0x00000000);

			err = send_cmd(mmc_base, SDMMC_ACMD41, ocr);
			if (err)
				return -err;

			ocr_rcvd =  readl(&mmc_base->rsp10);
		}

		if (ocr_rcvd & (1 << 31) == 0)
			return -1; /* card not ready yet */

		err = send_cmd(mmc_base, SDMMC_CMD2, ocr_rcvd);
		if (err)
			return -err;
	
	}

	if (cardtype == SD_CARD || cardtype == SDIO_CARD) {
		err = send_cmd(mmc_base, SDMMC_CMD3, 0x00000000); /* Get RCA */
		rca =  readl(&mmc_base->rsp10) >> 16;
	} else if (cardtype == MMC_CARD) {
		rca = 0x1234 << 16;
		err = send_cmd(mmc_base, MMC_CMD3, rca); /* Set RCA */
	}
	if (err)
		return -err;
	
	err = send_cmd(mmc_base, SDMMC_CMD7, rca << 16); /* set rca */
	if (err)
		return -err;

		/* The WL1271 does not respond to any higher freq than 4MHZ during ID reading Phase */
		/* Setting the clock to 2MHZ during the ID reading Phase as at higher frequency card does not respond */
	err = set_mmc_clock(mmc_base, CLK_24MHZ);
	if (err)
		return -err;

	return cardtype;
}

/*
 * Read a Block
 * 
 * Parameters:
 * mmc_base:			Base Address of the SD/SDIO controller
 * high_capacity_card:	1 if it is a High Capacity Card
 * block:				Block Address
 * nBytes:				size of the block
 * buf:					buffer to be read into
 * 
 * Returns:		
 *  < 0					On Error
 *   0					On Success	
 */ 
INT32 bread (hsmmc_t *mmc_base, INT32 high_capacity, \
				INT32 block_num, UINT32 nbytes, \
				UINT8 *buf)
{
	UINT32 arg;
	UINT32 status;
	INT32 err;
	INT32 i;

	if (high_capacity)
		arg = block_num; /* sector mode */
	else
		arg = block_num * SECTOR_SIZE; /* byte mode */

	err = send_cmd(mmc_base, SDMMC_CMD17, arg);
	if (err)
		return -err;
	
	do {
		status = readl(&mmc_base->stat);
	} while ( (status & STAT_BRR) == 0); /* ready to read? */

	for (i = 0; i < SECTOR_SIZE / 4; i++) {
		do {
			status = readl(&mmc_base->stat);
		} while ( (status & STAT_BRR) == 0); /* ready to read? */

		*(UINT32 *)buf = readl(&mmc_base->data);
		buf = buf + 4;
	}
	
	writel(readl(&mmc_base->stat) | STAT_BRR, &mmc_base->stat);
	
	return SUCCESS;
}

/*
 * Write a Block
 * 
 * Parameters:
 * mmc_base:			Base Address of the SD/SDIO controller
 * high_capacity_card:	1 if it is a High Capacity Card else, 0.
 * block:				Block Address
 * nBytes:				size of the block
 * buf:					data to be written
 * 
 * Returns:		
 *  < 0					On Error
 *   0					On Success	
 */ 
INT32 bwrite (hsmmc_t *mmc_base, INT32 high_capacity, \
			INT32 block_num, UINT32 nbytes, UINT8 *buf)
{
	UINT32 arg;
	UINT32 status;
	INT32 err;
	INT32 i;

	if (high_capacity)
		arg = block_num; /* sector mode */
	else
		arg = block_num * SECTOR_SIZE; /* byte mode */

	err = send_cmd(mmc_base, SDMMC_CMD24, arg);
	if (err)
		return -err;

	do {
		status = readl(&mmc_base->stat);
	} while ( (status & STAT_BWR) == 0); /* ready to write? */

	for (i = 0; i < SECTOR_SIZE / 4; i++) {
			writel( *(UINT32 *)buf, &mmc_base->data);
			buf = buf + 4;
	}
	
	writel(readl(&mmc_base->stat) | STAT_BWR, &mmc_base->stat);

	return SUCCESS;
}

INT32 get_sdio_chipid (hsmmc_t * mmc_base, UINT32 *vendor)
{
	
	UINT32 loop, cis_offset;
	UINT32 arg;
	INT32 err;

	cis_offset = 0;
	
	for(loop = 0xb; loop >= 0x9; loop--) {
		arg = (loop & 0x1FFFF) << 9; /* CIS Pointer Offset 9-0xB (3 Bytes) */	
		err = send_cmd(mmc_base, SDMMC_CMD52, arg);
 		if (err)
 			return -err;
 
		cis_offset = (cis_offset << 8) +  (readl(&mmc_base->rsp10) & 0xff);
	}
	
	for(loop = 0; loop < 50; loop++)
   	{
		arg = (cis_offset & 0x1FFFF) << 9;
		err = send_cmd(mmc_base, SDMMC_CMD52, arg);
		if (err)
 			return -err;
 			
 		if ( (readl(&mmc_base->rsp10) & 0xff) == 0x20) /* manfacturer tuple found */
      		break;
      
      // next tuple
 		cis_offset++;
		arg = (cis_offset & 0x1FFFF) << 9;

		err = send_cmd(mmc_base, SDMMC_CMD52, arg);
		if (err)
 			return -err;
 			
		cis_offset = cis_offset + (readl(&mmc_base->rsp10) & 0xff) + 1;
	}
   
	if (loop < 50)
	{
		// Manufacture tuple found
		cis_offset += 2;

		*vendor = 0;
		for(loop = 0; loop < 4; loop++)
		{
			arg = (cis_offset & 0x1FFFF) << 9;
			err = send_cmd(mmc_base, SDMMC_CMD52, arg);
			if (err)
				return -err;
				
			*vendor += ((readl(&mmc_base->rsp10) & 0xff)  << (loop * 8));
			cis_offset++;
		}
		
		return SUCCESS;	
   } else
   		platform_write ("\nUNABLE to read Manufacturer tuple\n");
   
   return FAILURE;
}
